"use strict";
var DeviceInfo = require("./DeviceInfo");
var fs = require('fs');
var progress = require('progress-stream');
var util = require('../util');
var ssh2 = require('ssh2');
var async = require('async');
var request = require('request');
var Device = (function () {
    function Device(di) {
        this.id = di._id;
        this.deviceInfo = di;
        this.address = di.address;
        this.ssh_port = di.ssh_port;
        this.web_port = di.web_port;
        this.username = di.username;
        this.password = di.password;
        this.monitor = di.monitor;
        this.name = di.name;
        this.model = di.model;
        this.firmware = di.firmware;
        this.uptime = -1;
        this.status = "";
        this.statusProgress = -1;
        this.bootid = '';
        this.bandwidth = 0;
        this.pps = 0;
        this.power = 0;
        this.temp = 0;
        this.web_protocol = '';
        this.https_port = 0;
        this.http_port = 0;
        this.warning = '';
        this.config_version = 0;
    }
    Device.prototype.query = function () {
        var device = this;
        if ('development' == global["app"].get('env') && this.address == "192.168.1.1") {
            // this is just for testing
            setTimeout(function () {
                device.model = "WS-12-250A";
                device.firmware = "1.3.9rc13";
                device.name = "Netonix Switch";
                device.status = "";
                device.statusProgress = -1;
                device.uptime = 666;
                device.bootid = "arandombootid";
                device.power = 8.1;
                device.pps = 3122;
                device.bandwidth = 1588;
                device.temp = 55;
                device.config_version = 1;
            }, 1000);
        }
        else {
            // its faster to get status updates via HTTP, so do that if possible (web server might be disabled on the switch)
            // but we have to use SSH initially to get the bootid
            if (device.bootid != '' && device.web_protocol != '') {
                if (device.monitor == false) {
                    device.status = '';
                    device.statusProgress = -1;
                    device.warning = '';
                    device.bandwidth = 0;
                    device.pps = 0;
                    device.power = 0;
                    device.temp = 0;
                    device.config_version = 0;
                    return;
                }
                var web_port;
                if (device.web_port != '') {
                    web_port = device.web_port;
                }
                else {
                    web_port = (device.web_protocol == 'https' ? device.https_port : device.http_port);
                }
                request({ url: device.web_protocol + "://" + device.address + ":" + web_port + "/api/v1/status/30sec?" + device.bootid, agentOptions: { rejectUnauthorized: false }, timeout: 2000 }, function (error, response, body) {
                    if (error) {
                        device.connectionError(error);
                    }
                    else if (response.statusCode == 200) {
                        updateDeviceStats(device, body);
                    }
                });
            }
            else {
                var conn = new ssh2();
                var commands = [
                    {
                        label: 'switch name',
                        command: 'grep Switch_Name /www/config.json |cut -d\\" -f4',
                        onclose: function (data) {
                            device.name = data.replace('\n', '');
                        }
                    },
                    {
                        label: 'bootid',
                        command: 'cat /tmp/bootid',
                        onclose: function (data) {
                            device.bootid = data.replace('\n', '');
                        }
                    },
                    {
                        label: 'web port',
                        command: 'grep HTTPS_Port /www/config.json |cut -d\\" -f4',
                        onclose: function (data) {
                            device.https_port = parseInt(data);
                        }
                    },
                    {
                        label: 'http port',
                        command: 'grep HTTPS_HTTP_Port /www/config.json |cut -d\\" -f4',
                        onclose: function (data) {
                            if (data == "") {
                                // firmware prior to 1.4.0 don't have this config value
                                device.http_port = 80;
                            }
                            else {
                                device.http_port = parseInt(data);
                            }
                        }
                    },
                    {
                        label: 'web protocol',
                        command: 'grep HTTPS_Allow_HTTP /www/config.json |cut -d: -f2',
                        onclose: function (data) {
                            if (data.indexOf("true") != -1) {
                                device.web_protocol = "http";
                            }
                            else {
                                device.web_protocol = "https";
                            }
                        }
                    },
                    {
                        label: 'web enabled',
                        command: 'grep HTTPS_Enable /www/config.json |cut -d: -f2',
                        onclose: function (data) {
                            if (data.indexOf("true") == -1) {
                                device.web_protocol = "";
                            }
                        }
                    },
                    {
                        label: '30sec.json',
                        command: 'cat /tmp/30sec.json',
                        onclose: function (data) {
                            if (device.monitor || device.model == '' || device.firmware == '') {
                                updateDeviceStats(device, data);
                            }
                        }
                    },
                ];
                conn.on('ready', function () {
                    var errorOccured = false;
                    async.eachSeries(commands, function (item, callback) {
                        try {
                            var buffer = '';
                            conn.exec(item.command, function (err, stream) {
                                if (err) {
                                    util.log(device.address + ": Error getting " + item.label);
                                    return callback(new Error(device.address + ": Error getting " + item.label));
                                }
                                stream.on('data', function (data) {
                                    buffer += data.toString();
                                });
                                stream.on('close', function () {
                                    if (item.onclose) {
                                        item.onclose(buffer);
                                    }
                                    callback();
                                });
                            });
                        }
                        catch (ex) {
                            errorOccured = true;
                            callback(ex);
                        }
                    }, function () {
                        // when all the commands are done
                        conn.end();
                        if (!errorOccured) {
                            DeviceInfo.update({ _id: device.id }, { name: device.name, model: device.model, firmware: device.firmware }, function (err, numAffected) {
                            });
                        }
                        else {
                            device.connectionError("SSH connection unexpected closed");
                        }
                    });
                }).on('error', function (err) {
                    if (err.message == 'All configured authentication methods failed') {
                        device.status = "Invalid username or password";
                    }
                    else {
                        device.connectionError("Unable to open SSH connection");
                    }
                }).connect({
                    host: device.deviceInfo.address,
                    port: device.deviceInfo.ssh_port,
                    username: device.deviceInfo.username,
                    password: device.deviceInfo.password,
                    readyTimeout: 5000
                });
            }
        }
    };
    Device.prototype.upgradeFirmware = function (firmwareFile, callback) {
        var device = this;
        // only start upgrade if the device is idle and online
        if (device.uptime > 0 && device.status == '') {
            util.log("Upgrading device " + device.address + " from " + device.firmware + " with firmware " + firmwareFile);
        }
        else {
            util.log("Not upgrading " + device.address + ", device is not available");
            if (callback)
                callback(null, false);
            return;
        }
        if ('development' == global["app"].get('env') && (this.address == "192.168.1.1" || this.address == "192.168.1.2")) {
            var n = 0;
            var timer = setInterval(function () {
                if (n == 0) {
                    device.status = "Uploading firmware";
                    device.statusProgress = 0;
                }
                else if (n == 1) {
                    device.statusProgress = 25;
                }
                else if (n == 2) {
                    device.statusProgress = 50;
                }
                else if (n == 3) {
                    device.statusProgress = 75;
                }
                else if (n == 4) {
                    device.status = "Upgrading firmware";
                    device.statusProgress = 0;
                }
                else if (n == 5) {
                    device.statusProgress = 25;
                }
                else if (n == 6) {
                    device.statusProgress = 50;
                }
                else if (n == 7) {
                    device.statusProgress = 75;
                }
                else if (n == 8) {
                    util.log("Upgrading device " + device.address + " complete, rebooting");
                    device.status = "Rebooting";
                    device.statusProgress = -1;
                    device.uptime = 0;
                    if (callback)
                        callback(null, true);
                }
                else if (n == 9) {
                    device.status = "";
                    device.statusProgress = -1;
                    device.uptime = 30;
                    clearInterval(timer);
                }
                n++;
            }, 1000);
        }
        else {
            var filename = global["datadir"] + '/firmwares/' + firmwareFile;
            if (!fs.existsSync(filename)) {
                util.notify("firmware file " + filename + " is missing", device);
                if (callback)
                    callback(null, false);
                return;
            }
            var Client = require('ssh2').Client;
            var conn = new Client();
            conn.on('ready', function () {
                conn.exec('scp -t /tmp/upgrade.bin', function (err, stream) {
                    if (err) {
                        util.notify("error copying firmware to remote", device);
                        if (callback)
                            callback(null, false);
                        return;
                    }
                    device.status = "Uploading firmware";
                    device.statusProgress = 0;
                    var stat = fs.statSync(filename);
                    stream.write("C0644 " + stat.size + " " + firmwareFile + "\n");
                    var str = progress({
                        length: stat.size,
                        time: 100
                    });
                    str.on('progress', function (progress) {
                        device.statusProgress = progress.percentage;
                    });
                    var rs = fs.createReadStream(filename);
                    rs.pipe(str).pipe(stream);
                    stream.on('close', function (data) {
                        var buffer = '';
                        var errorOccured = false;
                        function updateStatus(data) {
                            var text = (buffer + data.toString()).replace("\r\n", "\n");
                            var a = text.indexOf('\n');
                            while (a > -1) {
                                var msg = text.substr(0, a);
                                if (msg.indexOf("Error") > -1) {
                                    util.notify("error during upgrade: " + msg.replace("Error:", ""), device);
                                    errorOccured = true;
                                }
                                else if (msg.indexOf("...") > -1 || msg.indexOf("%") > -1) {
                                    if (msg.indexOf("%") > 0) {
                                        device.statusProgress = parseInt(msg);
                                    }
                                    else {
                                        device.status = msg;
                                        device.statusProgress = -1;
                                    }
                                }
                                text = text.substr(a + 1);
                                a = text.indexOf('\n');
                            }
                            buffer = text;
                            if (text.indexOf("...") > -1 || text.indexOf("%") > -1) {
                                if (text.indexOf("%") > 0) {
                                    device.statusProgress = parseInt(text);
                                }
                                else {
                                    device.status = text;
                                    device.statusProgress = -1;
                                }
                            }
                        }
                        conn.shell(function (err, stream) {
                            if (err) {
                                util.notify("error starting remote shell", device);
                                if (callback)
                                    callback(null, false);
                                return;
                            }
                            device.status = "Upgrading firmware";
                            device.statusProgress = 0;
                            stream.on('close', function () {
                                conn.end();
                                device.statusProgress = -1;
                                if (errorOccured) {
                                    device.status = "";
                                    if (callback)
                                        callback(null, false);
                                }
                                else {
                                    device.status = "Rebooting";
                                    util.log("Upgrading device " + device.address + " complete, rebooting");
                                    if (callback)
                                        callback(null, true);
                                }
                            }).on('data', function (data) {
                                updateStatus(data);
                            });
                            // calling firmware_upgrade like this makes it run in the background, so it will finish even if this session disconnects
                            stream.write('cmdline\n ((firmware_upgrade /tmp/upgrade.bin)&)&\n PID=`pgrep firmware_upgrad -o`\n kill -0 $PID 2>/dev/null\n while [ $? == 0 ]; do sleep 1; kill -0 $PID 2>/dev/null; done\nexit\nexit\n');
                        });
                    });
                });
            }).connect({
                host: device.deviceInfo.address,
                port: device.deviceInfo.ssh_port,
                username: device.deviceInfo.username,
                password: device.deviceInfo.password
            });
        }
    };
    Device.prototype.reboot = function () {
        var device = this;
        util.log("Rebooting device " + device.address);
        var conn = new ssh2();
        conn.on('ready', function () {
            conn.exec('reboot', function (err, stream) {
                if (err) {
                    util.log(device.address + ": Error rebooting");
                    return;
                }
                stream.on('close', function () {
                    device.status = "Rebooting";
                    conn.end();
                });
            });
        }).on('error', function (err) {
            if (err.message == 'All configured authentication methods failed') {
                device.status = "Invalid username or password";
            }
            else {
                device.connectionError("Unable to open SSH connection");
            }
        }).connect({
            host: device.deviceInfo.address,
            port: device.deviceInfo.ssh_port,
            username: device.deviceInfo.username,
            password: device.deviceInfo.password,
            readyTimeout: 5000
        });
    };
    Device.prototype.connectionError = function (msg) {
        var device = this;
        if (device.uptime > 0) {
            util.notify("changed state to Offline: " + msg, device);
            device.status = '';
            device.statusProgress = -1;
            device.warning = '';
            device.bootid = '';
            device.bandwidth = 0;
            device.pps = 0;
            device.power = 0;
            device.temp = 0;
            device.web_protocol = '';
            device.https_port = 0;
            device.http_port = 0;
            device.config_version = 0;
        }
        device.uptime = 0;
        device.status = '';
    };
    return Device;
}());
;
function updateDeviceStats(device, statsjson) {
    try {
        var stats = JSON.parse(statsjson);
        device.model = stats.Model;
        device.firmware = stats.Firmware_Version;
        if (device.monitor) {
            if (device.uptime == 0 && stats.Uptime > 0) {
                util.notify("changed state to Online", device);
                device.status = '';
            }
            device.uptime = stats.Uptime;
            device.temp = stats.Board_Temp;
            device.bandwidth = 0;
            device.bandwidth += stats.UsageData[0].txThroughputPerPort.reduce(function (a, b) { return a + b; });
            device.bandwidth += stats.UsageData[0].rxThroughputPerPort.reduce(function (a, b) { return a + b; });
            device.pps = 0;
            device.pps += stats.UsageData[0].txPacketRatePerPort.reduce(function (a, b) { return a + b; });
            device.pps += stats.UsageData[0].rxPacketRatePerPort.reduce(function (a, b) { return a + b; });
            device.power = stats.BasePowerUsage;
            device.power += stats.UsageData[0].powerPerPort.reduce(function (a, b) { return a + b; });
            if (stats.UnseenLogMessages) {
                device.warning = "Log alerts";
            }
            else {
                device.warning = '';
            }
            if (device.config_version != stats.Config_Version) {
                // the config changed, we need to do a full update
                device.bootid = '';
            }
            device.config_version = stats.Config_Version;
        }
    }
    catch (err) {
        console.log(device.address);
        console.log(statsjson);
    }
}
module.exports = Device;
//# sourceMappingURL=Device.js.map