"use strict";
var express = require('express');
var http = require('http');
var https = require('https');
var path = require('path');
var fs = require('fs');
var async = require('async');
var tungus = require('tungus');
var mongoose = require('mongoose');
var cidr = require('cidr-range');
var util = require('./util');
var multer = require('multer');
// express middleware
var request = require('request');
var progress = require('progress-stream');
var expressLogger = require('morgan');
var bodyParser = require('body-parser');
var methodOverride = require('method-override');
var engine = require('ejs-mate');
var cookieParser = require('cookie-parser');
var cookieSession = require('cookie-session');
var app = express();
global["app"] = app;
// import models
var Config = require("./models/Config");
var Device = require("./models/Device");
var DeviceInfo = require("./models/DeviceInfo");
var Log = require("./models/Log");
var Schedule = require("./models/Schedule");
var devices = {};
var repositoryFirmwares = [];
var version = fs.readFileSync("version.txt", "utf8").replace('\r', '').replace('\n', '');
if ('development' == app.get('env')) {
    global["datadir"] = __dirname;
}
else {
    if (process.platform == 'darwin') {
        global["datadir"] = process.env.HOME + 'Library/Preferences/Netonix Manager';
    }
    else if (process.platform == 'win32') {
        global["datadir"] = process.env.APPDATA + "/Netonix Manager";
    }
    else {
        // linux
        global["datadir"] = '/var/lib/netonix-manager';
    }
    if (fs.existsSync(global["datadir"]) == false) {
        fs.mkdirSync(global["datadir"]);
    }
    if (fs.existsSync(global["datadir"] + "/firmwares") == false) {
        fs.mkdirSync(global["datadir"] + "/firmwares");
    }
    if (fs.existsSync(global["datadir"] + "/storage") == false) {
        fs.mkdirSync(global["datadir"] + "/storage");
    }
}
var storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, path.join(global["datadir"], 'firmwares'));
    },
    filename: function (req, file, cb) {
        cb(null, file.originalname);
    }
});
// start up DB
var db = mongoose.connection;
db.on('error', console.error);
mongoose.connect('tingodb://' + global["datadir"] + '/storage', function (err) {
    util.log("Version " + version + " starting up");
    // create default configuration if there is no configuration
    Config.findOne({}, function (err, config) {
        if (err)
            return console.log(err);
        if (config == null) {
            config = new Config();
            config.username = "admin";
            config.password = "admin";
            config.last_seen_version = version;
            config.save();
        }
    });
});
var sessionToken = "";
require('crypto').randomBytes(48, function (err, buffer) {
    sessionToken = buffer.toString('hex');
});
// all environments
app.engine('ejs', engine);
app.set('port', process.env.PORT || 3000);
app.set('views', path.join(__dirname, 'views'));
app.set('view engine', 'ejs');
app.use(expressLogger('dev'));
app.use(require('express-domain-middleware'));
app.use(cookieParser());
app.use(cookieSession({ name: 'session', secret: 'thisisnoterysecret' }));
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));
app.use(methodOverride());
app.use(express.static(path.join(__dirname, 'public')));
app.use(function (req, res, next) {
    // we use this for testing
    if ('development' == app.get('env') && req.header("Authorization") == "lasdhsvilnslinvgsdgf") {
        next();
        return;
    }
    if (req.url !== '/' && (!req.session || req.session.token != sessionToken)) {
        res.sendStatus(401);
        return;
    }
    next();
});
// routes
app.get('/', function (req, res) {
    var tarpit_delay = 0;
    if (req.session.tarpit_time) {
        req.session.tarpit_time = new Date(req.session.tarpit_time);
        if (new Date() >= req.session.tarpit_time) {
            delete req.session.tarpit_time;
            delete req.session.loginFailureCount;
        }
        else {
            tarpit_delay = Math.ceil((req.session.tarpit_time.getTime() - new Date().getTime()) / 1000);
        }
    }
    res.render('index', { title: 'Netonix Manager', _layoutFile: 'layout', error: '', tarpit_delay: tarpit_delay });
});
app.post('/', function (req, res, next) {
    Config.findOne({}, function (err, config) {
        if (err)
            return console.log(err);
        if (req.body.username && req.body.username === config.username && req.body.password && config.comparePassword(req.body.password)) {
            req.session.token = sessionToken;
            if (repositoryFirmwares.length == 0) {
                refreshFirmwareList();
            }
            res.redirect('/main.html');
        }
        else {
            var tarpit_delay = 0;
            util.log("Invalid login attempt for user '" + req.body.username + "'");
            if (req.session.loginFailureCount) {
                req.session.loginFailureCount++;
                if (req.session.loginFailureCount >= 5) {
                    req.session.tarpit_time = new Date(new Date().getTime() + 60000);
                    tarpit_delay = 60;
                }
            }
            else {
                req.session.loginFailureCount = 1;
            }
            res.render('index', { title: 'Netonix Manager', _layoutFile: 'layout', error: 'Invalid username or password', tarpit_delay: tarpit_delay });
        }
    });
});
app.get('/logout', function (req, res, next) {
    delete req.session.token;
    res.redirect('/');
});
app.get('/api/version', function (req, res) {
    res.json({ version: version, crash_log: fs.existsSync("crash.log") });
});
app.post('/api/crashlog/:op', function (req, res) {
    var logdata = fs.readFileSync("crash.log");
    fs.unlinkSync("crash.log");
    request.post('http://forum.netonix.com/netonix_manager_crash_report.php', { form: { message: logdata, submit: 'submit', version: version } }, function (error, response, body) {
    });
    res.json({});
});
function getDevice(device) {
    var newObject = JSON.parse(JSON.stringify(device));
    delete newObject.deviceInfo;
    return newObject;
}
app.get('/api/devices', function (req, res) {
    var result = Array();
    for (var key in devices) {
        result.push(getDevice(devices[key]));
    }
    res.json({ devices: result });
});
app.post('/api/devices', function (req, res) {
    DeviceInfo.findOne({ address: req.body.device.address, ssh_port: req.body.device.ssh_port }, function (err, device) {
        if (device == null) {
            DeviceInfo.create(req.body.device, function (err, device) {
                if (err) {
                    // this will happen if the record already exists
                    res.sendStatus(409);
                }
                else {
                    devices[device.id] = new Device(device);
                    res.redirect("/api/devices/" + device.id);
                    devices[device.id].query();
                    util.log("Created device " + device.address);
                }
            });
        }
        else {
            // this will happen if the device already exists
            res.sendStatus(409);
        }
    });
});
app.get('/api/devices/:id', function (req, res) {
    res.json(getDevice(devices[req.params.id]));
});
app.put('/api/devices/:id', function (req, res) {
    DeviceInfo.update({ _id: req.params.id }, req.body.device, function (err, numAffected) {
        if (err) {
            // if the id doesn't exist
            res.sendStatus(404);
        }
        else {
            DeviceInfo.findById(req.params.id, function (err, deviceinfo) {
                devices[req.params.id] = new Device(deviceinfo);
                devices[req.params.id].query();
                util.log("Updated device " + devices[req.params.id].address);
                res.sendStatus(204); // success, no content
            });
        }
    });
});
app.delete('/api/devices/:id', function (req, res) {
    DeviceInfo.remove({ _id: req.params.id }, function (err) {
        if (err) {
            // if the id doesn't exist
            res.sendStatus(404);
        }
        else {
            util.log("Deleted device " + devices[req.params.id].address);
            delete devices[req.params.id];
            res.sendStatus(204); // success, no content
        }
    });
});
app.get('/api/firmwares', function (req, res) {
    res.json({ firmwares: fs.readdirSync(path.join(global["datadir"], 'firmwares')) });
});
app.delete('/api/firmwares/:id', function (req, res) {
    fs.unlinkSync(path.join(path.join(global["datadir"], 'firmwares'), req.params.id));
    res.sendStatus(204); // success, no content
});
app.get('/api/repositoryfirmwares', function (req, res) {
    res.json({ firmwares: repositoryFirmwares });
});
app.post('/api/uploadfirmware', multer({ storage: storage, limits: { fileSize: 10000000, files: 1 } }).single('file'), function (req, res) {
    res.sendStatus(204).end();
});
function upgradeDevices(body, firmwareFile) {
    if (body.scheduled) {
        var s = new Schedule();
        s.date = body.date;
        s.firmware = "Upgrade to " + firmwareFile;
        s.devices = body.devices;
        s.save();
    }
    else {
        body.devices.forEach(function (id) {
            if (devices[id] != null) {
                devices[id].upgradeFirmware(firmwareFile, null);
            }
        });
    }
}
function scanAddress(address, deviceModel) {
    DeviceInfo.findOne({ address: address, ssh_port: deviceModel.ssh_port }, function (err, device) {
        if (device == null) {
            // very early firmwares used HTTP Basic Authentication so we need to specify the username/password
            request({ url: 'https://' + deviceModel.username + ':' + deviceModel.password + '@' + address, agentOptions: { rejectUnauthorized: false } }, function (error, response, body) {
                if (!error && response.statusCode == 200) {
                    if (body.indexOf('<i>WISP</i> Switch') > 0 || body.indexOf('Netonix') > 0) {
                        deviceModel.address = address;
                        DeviceInfo.create(deviceModel, function (err, device) {
                            if (!err) {
                                util.log("Scan found new device at " + address);
                                devices[device.id] = new Device(device);
                                devices[device.id].query();
                            }
                        });
                    }
                }
            });
        }
    });
}
function netmask2cidr(mask) {
    var maskNodes = mask.match(/(\d+)/g);
    var cidr = 0;
    for (var i in maskNodes) {
        cidr += (((maskNodes[i] >>> 0).toString(2)).match(/1/g) || []).length;
    }
    return cidr;
}
app.post('/api/scanrange', function (req, res) {
    var range = req.body.device.address;
    util.log("Scanning " + range + " for devices");
    var s = range.split('/');
    if (s[1].indexOf('.') > 0) {
        range = s[0] + '/' + netmask2cidr(s[1]);
    }
    var ips = cidr(range, { onlyHosts: true });
    var tasks = [];
    ips.forEach(function (val) {
        tasks.push(function () {
            scanAddress(val, req.body.device);
        });
    });
    async.parallel(tasks);
    res.sendStatus(204).end();
});
app.post('/api/reboot', function (req, res) {
    var body = req.body;
    if (body.scheduled) {
        var s = new Schedule();
        s.date = body.date;
        s.firmware = "Reboot";
        s.devices = body.devices;
        s.save();
    }
    else {
        body.devices.forEach(function (id) {
            if (devices[id] != null) {
                devices[id].reboot();
            }
        });
    }
    res.sendStatus(204); // success, no content
});
function setDeviceStatus(req, msg, progress) {
    req.body.devices.forEach(function (id) {
        if (devices[id] != null) {
            devices[id].status = msg;
            devices[id].statusProgress = progress;
        }
    });
}
function doDownload(req, url, errorcb) {
    var localFilename = global["datadir"] + '/firmwares/' + req.body.downloadFile;
    if (fs.existsSync(localFilename)) {
        util.log(req.body.downloadFile + " already exists, skipping download");
        upgradeDevices(req.body, req.body.downloadFile);
        return;
    }
    util.log("Downloading " + req.body.downloadFile);
    var file = fs.createWriteStream(localFilename);
    var request = http.get(url, function (response) {
        if (response.statusCode == 404) {
            file.close();
            fs.unlinkSync(localFilename);
            setDeviceStatus(req, '', -1);
            errorcb();
            return;
        }
        var str = progress({
            length: response.headers["content-length"],
            time: 100
        });
        str.on('progress', function (progress) {
            setDeviceStatus(req, "Downloading " + req.body.downloadFile, progress.percentage);
        });
        response.pipe(str).pipe(file);
        file.on('finish', function () {
            file.close();
            setDeviceStatus(req, '', -1);
            upgradeDevices(req.body, req.body.downloadFile);
        });
    }).on('error', function (err) {
        file.close();
        fs.unlinkSync(localFilename);
        setDeviceStatus(req, '', -1);
        errorcb();
    });
}
app.post('/api/upgradefirmware', function (req, res) {
    if (req.body.downloadFile) {
        doDownload(req, 'http://forum.netonix.com/firmware/' + req.body.downloadFile, function () {
            // download failed, maybe the file is in the archive directory (thanks for the typo Chris)
            doDownload(req, 'http://forum.netonix.com/firmware/archieve/' + req.body.downloadFile, function () {
                // not there either, give up
                util.log("Unable to download " + req.body.downloadFile);
            });
        });
    }
    else {
        upgradeDevices(req.body, req.body.firmwareFile);
    }
    res.sendStatus(204).end();
});
app.post('/api/testsmtp', function (req, res) {
    util.sendEmail('test email from Netonix Manager', 'This is a test email', req.body, function (errorMessage) {
        res.json({ error: errorMessage });
    });
});
app.get('/api/settings', function (req, res) {
    Config.findOne({}, function (err, config) {
        if (err)
            return console.log(err);
        var settings = JSON.parse(JSON.stringify(config));
        // for security reasons we don't want to send this
        delete settings._id;
        delete settings.password;
        res.json(settings);
    });
});
app.put('/api/settings', function (req, res) {
    Config.findOne({}, function (err, config) {
        // we can't use Config.update() here because we need to pre-save method to be called
        for (var key in config) {
            var s = req.body[key];
            if (typeof (s) !== 'undefined') {
                config[key] = req.body[key];
                // this is for testing
                if (key == 'last_seen_version') {
                    version = req.body[key];
                }
            }
        }
        config.save();
        util.log("Updated settings");
        res.sendStatus(204); // success, no content
    });
});
app.get('/api/log', function (req, res) {
    Log.find({}, function (err, results) {
        res.json({ log: results });
    });
});
app.post('/api/log', function (req, res) {
    Log.create(req.body.log, function (err, newlog) {
        if (err) {
            // this will happen if the record already exists
            res.sendStatus(409);
        }
        else {
            res.redirect("/api/log");
        }
    });
});
app.delete('/api/log', function (req, res) {
    Log.remove({}, function (err) {
        res.sendStatus(204); // success, no content
    });
});
app.get('/api/schedules', function (req, res) {
    Schedule.find({}, function (err, results) {
        res.json({ schedules: results });
    });
});
app.get('/api/versioncheck', function (req, res) {
    refreshFirmwareList();
    res.sendStatus(204); // success, no content
});
app.delete('/api/schedules/:id', function (req, res) {
    Schedule.remove({ _id: req.params.id }, function (err) {
        if (err) {
            // if the id doesn't exist
            res.sendStatus(404);
        }
        else {
            res.sendStatus(204); // success, no content
        }
    });
});
var updateStatus = "";
var updateProgress = 0;
app.get('/api/update', function (req, res) {
    res.send({ status: updateStatus, progress: updateProgress });
});
app.post('/api/update', function (req, res) {
    updateStatus = "Downloading";
    var localFilename = './download.tar.gz';
    var file = fs.createWriteStream(localFilename);
    var request = http.get('http://forum.netonix.com/netonix-manager/netonix-manager-' + req.body.version + ".tar.gz", function (response) {
        if (response.statusCode == 404) {
            file.close();
            fs.unlinkSync(localFilename);
            res.sendStatus(404);
            return;
        }
        res.sendStatus(204); // success, no content
        var str = progress({
            length: response.headers['content-length'],
            time: 100
        });
        str.on('progress', function (progress) {
            updateProgress = progress.percentage;
        });
        response.pipe(str).pipe(file);
        file.on('finish', function () {
            file.close();
            fs.rename(localFilename, './update.tar.gz');
            updateStatus = "Updating";
            // the bootloader should now notice that update.tar.gz exists and begin the update, which will kill this process
        });
    }).on('error', function (err) {
        file.close();
        fs.unlinkSync(localFilename);
        res.sendStatus(404);
    });
});
// start refreshing all existing devices
setInterval(function () {
    for (var key in devices) {
        var device = devices[key];
        device.query();
    }
}, 5000);
DeviceInfo.find({}, function (err, deviceinfos) {
    if (err) {
    }
    else {
        deviceinfos.forEach(function (deviceinfo) {
            devices[deviceinfo.id] = new Device(deviceinfo);
            devices[deviceinfo.id].query();
        });
    }
});
function refreshFirmwareList() {
    Config.findOne({}, function (err, config) {
        request('http://forum.netonix.com/firmware', function (error, response, body) {
            if (error) {
                util.log("Error getting list of firmwares from the Netonix server");
            }
            else if (response.statusCode == 200) {
                repositoryFirmwares = [];
                var myArray;
                var re = /<a href="(wispswitch-.*bin)">/g;
                while ((myArray = re.exec(body)) !== null) {
                    repositoryFirmwares.push(myArray[1]);
                }
                // sort by version descending
                util.sortVersions(repositoryFirmwares);
                for (var i = 0; i < repositoryFirmwares.length; i++) {
                    if (repositoryFirmwares[i].indexOf("rc") == -1) {
                        var ver = repositoryFirmwares[i].replace("wispswitch-", "").replace(".bin", "");
                        if (ver != config.latest_firmware) {
                            if (config.latest_firmware != '') {
                                util.notify("New firmware version available: " + ver);
                            }
                            config.latest_firmware = ver;
                            config.save();
                        }
                        break;
                    }
                }
            }
        });
        setTimeout(function () {
            request('http://forum.netonix.com/netonix-manager', function (error, response, body) {
                if (error) {
                    util.log("Error getting list of Netonix Managers from the Netonix server");
                }
                else if (response.statusCode == 200) {
                    var nms = [];
                    var myArray;
                    var re = /<a href="netonix-manager-(.*)\.exe">/g;
                    while ((myArray = re.exec(body)) !== null) {
                        nms.push(myArray[1]);
                    }
                    if ('development' == app.get('env')) {
                        nms = ["1.0.1"];
                    }
                    util.sortVersions(nms);
                    if (nms[0] != config.last_seen_version) {
                        if (config.last_seen_version != '' && nms[0] != version) {
                            util.notify("New Netonix Manager version available: " + nms[0]);
                        }
                        config.last_seen_version = nms[0];
                        config.save();
                    }
                }
            });
        }, 500);
        // expire old log entries
        var d = new Date();
        d.setDate(d.getDate() - config.log_expire_days);
        Log.find({ timestamp: { $lt: d } }).remove().exec();
    });
}
setInterval(refreshFirmwareList, 3600000); // once an hour
setInterval(function () {
    Schedule.find({}, function (err, results) {
        var now = new Date();
        results.forEach(function (schedule) {
            if (schedule.date <= now) {
                util.log("Starting scheduled operation");
                if (schedule.firmware == "Reboot") {
                    schedule.devices.forEach(function (id) {
                        if (devices[id] != null) {
                            devices[id].reboot();
                        }
                    });
                    util.notify("Scheduled reboot complete, " + results.length + " devices rebooted");
                }
                else {
                    var upgrades = [];
                    var firmwareFile = schedule.firmware.replace("Upgrade to ", "");
                    schedule.devices.forEach(function (id) {
                        if (devices[id] != null) {
                            upgrades.push(function (callback) {
                                devices[id].upgradeFirmware(firmwareFile, callback);
                            });
                        }
                    });
                    async.parallel(upgrades, function (err, results) {
                        var successCount = results.filter(function (x) { return x == true; }).length;
                        util.notify("Scheduled upgrade complete, " + successCount + " of " + results.length + " devices successfully upgraded");
                    });
                }
                schedule.remove();
            }
        });
    });
}, 1000);
// error handlers need to be last in the handler chain
function errorHandler(err, req, res, next) {
    console.log('error on request %s %s: %j', req.method, req.url, err);
    util.log('Error in ' + req.method + ' ' + req.url + ': ' + err);
    try {
        res.status(500).send("Something bad happened. :(");
    }
    catch (ex) {
    }
}
app.use(errorHandler);
if ('development' == app.get('env')) {
    http.createServer(app).listen(app.get('port'), function () {
        console.log('Express server listening on port ' + app.get('port'));
    });
}
else {
    https.createServer({ key: fs.readFileSync('./private.key'), cert: fs.readFileSync('./certificate.pem') }, app).listen(3443, function () {
        console.log('Express server listening on port 3443');
    });
}
//# sourceMappingURL=app.js.map